/* Copyright (C) 2016-2018 RealVNC Ltd.  All Rights Reserved.
 */

/* This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

#include "VNCCommonDecoderPluginContextImpl.h"

#include "BaseDecoderUtils.h"

namespace
{
    bool convertVncBoolToBool(const vnc_bool_t value)
    {
        if(value == vnc_true)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    vnc_bool_t convertBoolToVncBool(const bool value)
    {
        if(value)
        {
            return vnc_true;
        }
        else
        {
            return vnc_false;
        }
    }
}

VNCCommonDecoderPluginContextImpl::VNCCommonDecoderPluginContextImpl(
        const VNCCommonDecoderSupportingAPI& decoderSupportingAPI,
        VNCCommonDecoderFrameworkContext decoderFrameworkContext)

    :   mSupportingApi(decoderSupportingAPI),
        mLogHandler(mSupportingApi, decoderFrameworkContext),
        mLog(mLogHandler.tag("BaseDecoder")),
        mSupport(mSupportingApi, decoderFrameworkContext, mLogHandler),
        mImpl()
{
}

BaseDecoderSupport& VNCCommonDecoderPluginContextImpl::support()
{
    return mSupport;
}

void VNCCommonDecoderPluginContextImpl::setImpl(
        std::auto_ptr<BaseDecoderImpl> impl)
{
    mImpl.reset(impl);
}

void VNCCommonDecoderPluginContextImpl::callbackDestroy()
{
    if(!mImpl.hasValue())
    {
        mLog.error("BaseDecoderImpl not found in callbackDestroy()");
        return;
    }

    mImpl.value().destroy();
    mImpl.reset();
}

#define CALLBACK_START_VOID \
        if(!mImpl.hasValue()) \
        { \
            mLog.error("BaseDecoderImpl not found in %s", __FUNCTION__); \
            mSupport.supportingApiError(VNCCommonDecoderErrorInternal); \
            return; \
        } \
        try {

#define CALLBACK_START_ERROR \
        if(!mImpl.hasValue()) \
        { \
            mLog.error("BaseDecoderImpl not found in %s", __FUNCTION__); \
            return VNCCommonDecoderErrorInternal; \
        } \
        try {

#define CALLBACK_END_VOID \
        } catch(const std::exception& e) { \
            mLog.error("Unhandled exception in %s: '%s'", __FUNCTION__, e.what()); \
            mSupport.supportingApiError(VNCCommonDecoderErrorInternal); \
        } catch(...) { \
            mLog.error("Unhandled exception in %s. Unknown exception type.", __FUNCTION__); \
            mSupport.supportingApiError(VNCCommonDecoderErrorInternal); \
        }

#define CALLBACK_END_ERROR \
        } catch(const std::exception& e) { \
            mLog.error("Unhandled exception in %s: '%s'", __FUNCTION__, e.what()); \
            return VNCCommonDecoderErrorInternal; \
        } catch(...) { \
            mLog.error("Unhandled exception in %s. Unknown exception type.", __FUNCTION__); \
            return VNCCommonDecoderErrorInternal; \
        }

#define CALLBACK_CHECK_NOT_NULL_VOID(x) \
        if(!x) \
        { \
            mLog.error("NULL '%s' in %s", #x, __FUNCTION__); \
            mSupport.supportingApiError(VNCCommonDecoderErrorInvalidArgument); \
            return; \
        }

#define CALLBACK_CHECK_NOT_NULL_ERROR(x) \
        if(!x) \
        { \
            mLog.error("NULL '%s' in %s", #x, __FUNCTION__); \
            return VNCCommonDecoderErrorInvalidArgument; \
        }

void VNCCommonDecoderPluginContextImpl::callbackStreamCreatedImpl(
        const vnc_uint64_t streamId,
        const VNCCommonDecoderMediaType mediaType,
        const VNCCommonDecoderStreamSpecificType specificType)
{
    CALLBACK_START_VOID

    mImpl.value().streamCreated(
            StreamID(streamId),
            mediaType,
            specificType);

    CALLBACK_END_VOID
}

VNCCommonDecoderError VNCCommonDecoderPluginContextImpl::callbackQueryEncodingSupportAudioLPCMImpl(
        const VNCCommonDecoderAudioFormatDetailsLPCM* formatDetails,
        vnc_bool_t *const out_isEncodingSupported,
        VNCCommonDecoderAudioFormatDetailsLPCM **const out_suggestedFormatDetailsList,
        vnc_size_t *const out_suggestedFormatDetailsCount)
{
    CALLBACK_START_ERROR
    CALLBACK_CHECK_NOT_NULL_ERROR(formatDetails)

    bool isEncodingSupported = false;
    mSupportedAudioFormatsResponse.clear();

    const VNCCommonDecoderError result
            = mImpl.value().querySupportAudioLPCM(
                    *formatDetails,
                    isEncodingSupported, // Reference
                    mSupportedAudioFormatsResponse);

    *out_isEncodingSupported = convertBoolToVncBool(isEncodingSupported);

    *out_suggestedFormatDetailsCount
            = mSupportedAudioFormatsResponse.size();

    if(mSupportedAudioFormatsResponse.size() > 0)
    {
        *out_suggestedFormatDetailsList
                = &(mSupportedAudioFormatsResponse[0]);
    }
    else
    {
        *out_suggestedFormatDetailsList = NULL;
    }

    return result;

    CALLBACK_END_ERROR
}

void VNCCommonDecoderPluginContextImpl::callbackStreamAudioContentHintImpl(
        const vnc_uint64_t streamId,
        const VNCCommonDecoderAudioStreamContentType* audioStreamContentTypeList,
        const vnc_size_t audioStreamContentTypeCount)
{
    CALLBACK_START_VOID

    std::vector<VNCCommonDecoderAudioStreamContentType> contentTypes;

    if(audioStreamContentTypeCount > 0)
    {
        CALLBACK_CHECK_NOT_NULL_VOID(audioStreamContentTypeList)

        contentTypes.assign(
                audioStreamContentTypeList,
                audioStreamContentTypeList + audioStreamContentTypeCount);
    }

    mImpl.value().streamHintAudioContent(
            StreamID(streamId),
            contentTypes);

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackStreamAudioEncodingHintLPCMImpl(
        const vnc_uint64_t streamId,
        const VNCCommonDecoderAudioFormatDetailsLPCM *const formatDetails)
{
    CALLBACK_START_VOID
    CALLBACK_CHECK_NOT_NULL_VOID(formatDetails)

    mImpl.value().streamHintAudioEncodingLPCM(
            StreamID(streamId),
            *formatDetails);

    CALLBACK_END_VOID
}

VNCCommonDecoderError
VNCCommonDecoderPluginContextImpl::callbackQueryEncodingSupportVideoH264Impl(
        const vnc_size_t h264AttributesSize,
        const VNCCommonDecoderH264Attributes* h264Attributes,
        const VNCCommonDecoderVideoMode* videoModeList,
        const vnc_size_t videoModeCount,
        vnc_bool_t *const out_isEncodingSupportedList,
        VNCCommonDecoderVideoMode* *const out_suggestedVideoModeList,
        vnc_size_t *const out_suggestedVideoModeCount)
{
    CALLBACK_START_ERROR
    CALLBACK_CHECK_NOT_NULL_ERROR(h264Attributes)
    CALLBACK_CHECK_NOT_NULL_ERROR(out_suggestedVideoModeList)
    CALLBACK_CHECK_NOT_NULL_ERROR(out_suggestedVideoModeCount)

    std::vector<VNCCommonDecoderVideoMode> videoModes;
    std::vector<bool> isVideoModeSupported(videoModeCount, false);

    if(videoModeCount > 0)
    {
        CALLBACK_CHECK_NOT_NULL_ERROR(videoModeList)
        CALLBACK_CHECK_NOT_NULL_ERROR(out_isEncodingSupportedList)

        videoModes.assign(
                videoModeList,
                videoModeList + videoModeCount);
    }

    mSupportedVideoFormatsResponse.clear();

    if(isVideoModeSupported.size() != videoModeCount)
    {
        mLog.error("Encoding query invalid argument: invalid output length");
        return VNCCommonDecoderErrorInvalidArgument;
    }

    const VNCCommonDecoderError result
            = mImpl.value().querySupportVideoH264(
                    h264AttributesSize,
                    *h264Attributes,
                    videoModes,
                    isVideoModeSupported,
                    mSupportedVideoFormatsResponse);

    if(isVideoModeSupported.size() != videoModeCount)
    {
        mLog.error(
                "Encoding query internal error: result output length changed");
        return VNCCommonDecoderErrorInternal;
    }

    for(size_t i = 0; i < videoModeCount; i++)
    {
        out_isEncodingSupportedList[i]
                = convertBoolToVncBool(isVideoModeSupported[i]);
    }

    *out_suggestedVideoModeCount = mSupportedVideoFormatsResponse.size();

    if(mSupportedVideoFormatsResponse.size() > 0)
    {
        *out_suggestedVideoModeList
                = &(mSupportedVideoFormatsResponse[0]);
    }

    return result;

    CALLBACK_END_ERROR
}

void VNCCommonDecoderPluginContextImpl::callbackStreamVideoEncodingHintH264Impl(
        const vnc_uint64_t streamId,
        const vnc_size_t h264AttributesSize,
        const VNCCommonDecoderH264Attributes *const h264Attributes,
        const VNCCommonDecoderVideoMode *const videoMode)
{
    CALLBACK_START_VOID
    CALLBACK_CHECK_NOT_NULL_VOID(h264Attributes)

    mImpl.value().streamHintVideoEncodingH264(
            StreamID(streamId),
            h264AttributesSize,
            *h264Attributes,
            videoMode
                    ? vnccommon::MakeOptional(*videoMode)
                    : vnccommon::Optional<VNCCommonDecoderVideoMode>());

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackStreamStartedImpl(
        const vnc_uint64_t streamId)
{
    CALLBACK_START_VOID

    mImpl.value().streamStarted(StreamID(streamId));

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackStreamStoppedImpl(
        const vnc_uint64_t streamId,
        const vnc_bool_t stopImmediately)
{
    CALLBACK_START_VOID

    mImpl.value().streamStopped(
            StreamID(streamId),
            convertVncBoolToBool(stopImmediately));

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackStreamDestroyedImpl(
        const vnc_uint64_t streamId)
{
    CALLBACK_START_VOID

    mImpl.value().streamDestroyed(StreamID(streamId));

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackStreamPayloadImpl(
        const vnc_uint64_t streamId,
        const vnc_size_t payloadStructSize,
        VNCCommonDecoderStreamPayload *const payload)
{
    CALLBACK_START_VOID

    mImpl.value().streamPayload(
            StreamID(streamId),
            payloadStructSize,
            payload);

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackStreamMuteEnableImpl(
        const vnc_uint64_t streamId)
{
    CALLBACK_START_VOID

    mImpl.value().streamMuteEnable(StreamID(streamId));

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackStreamMuteDisableImpl(
        const vnc_uint64_t streamId)
{
    CALLBACK_START_VOID

    mImpl.value().streamMuteDisable(StreamID(streamId));

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackStreamDuckEnableImpl(
        const vnc_uint64_t streamId,
        const vnc_uint64_t suggestedRampMs,
        const vnc_uint32_t suggestedAttenuationDb)
{
    CALLBACK_START_VOID

    mImpl.value().streamDuckEnable(
            StreamID(streamId),
            suggestedRampMs > 0
                    ? vnccommon::MakeOptional(suggestedRampMs)
                    : vnccommon::Optional<vnc_uint64_t>(),
            suggestedAttenuationDb > 0
                    ? vnccommon::MakeOptional(suggestedAttenuationDb)
                    : vnccommon::Optional<vnc_uint32_t>());

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackStreamDuckDisableImpl(
        const vnc_uint64_t streamId)
{
    CALLBACK_START_VOID

    mImpl.value().streamDuckDisable(StreamID(streamId));

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackStreamFrameRateLimitRequestSuccessImpl(
        const vnc_uint64_t streamId,
        const vnc_uint32_t newFrameRate)
{
    CALLBACK_START_VOID

    mImpl.value().streamNotifyFramerateLimitRequestSuccess(
            StreamID(streamId),
            newFrameRate > 0
                    ? vnccommon::MakeOptional(newFrameRate)
                    : vnccommon::Optional<vnc_uint32_t>());

    CALLBACK_END_VOID
}

VNCCommonDecoderError VNCCommonDecoderPluginContextImpl::callbackPropertySetImpl(
        const VNCCommonDecoderPropertyKey propertyKey,
        const VNCCommonDecoderDataType propertyType,
        void *const propertyValue)
{
    CALLBACK_START_ERROR

    switch(propertyType)
    {
        case VNCCommonDecoderDataTypeBoolean:
        {
            CALLBACK_CHECK_NOT_NULL_ERROR(propertyValue);

            const bool value = convertVncBoolToBool(
                    *static_cast<vnc_bool_t*>(propertyValue));

            return mImpl.value().setPropertyBool(
                    propertyKey,
                    value);
        }

        case VNCCommonDecoderDataTypeRectangle:
        {
            CALLBACK_CHECK_NOT_NULL_ERROR(propertyValue);

            return mImpl.value().setPropertyRectangle(
                    propertyKey,
                    *static_cast<VNCRectangle*>(propertyValue));
        }

        case VNCCommonDecoderDataTypeString:
        {
            CALLBACK_CHECK_NOT_NULL_ERROR(propertyValue);

            return mImpl.value().setPropertyString(
                    propertyKey,
                    std::string(
                            static_cast<const char*>(propertyValue)));
        }

        default:
        {
            return mImpl.value().setPropertyGeneric(
                    propertyKey,
                    propertyType,
                    propertyValue);
        }
    }

    CALLBACK_END_ERROR
}

void VNCCommonDecoderPluginContextImpl::callbackGetEventHandlesImpl(
        VNCCommonDecoderEventHandleParams **const out_eventHandleList,
        vnc_size_t *const out_eventHandleCount)
{
    CALLBACK_START_VOID

    // Just in case an exception is thrown
    *out_eventHandleCount = 0;

    mEventHandlesResponse.clear();

    mImpl.value().getEventHandles(
            mEventHandlesResponse);

    *out_eventHandleCount = mEventHandlesResponse.size();

    if(mEventHandlesResponse.size() > 0)
    {
        *out_eventHandleList = &(mEventHandlesResponse[0]);
    }
    else
    {
        *out_eventHandleList = NULL;
    }

    CALLBACK_END_VOID
}

void VNCCommonDecoderPluginContextImpl::callbackEventHandleActivityImpl(
        const VNCCommonDecoderEventHandle *const activeEventHandleList,
        const size_t activeEventHandleCount)
{
    CALLBACK_START_VOID

    mActiveEventHandles.resize(0);

    mActiveEventHandles.insert(
            mActiveEventHandles.end(),
            activeEventHandleList,
            activeEventHandleList + activeEventHandleCount);

    mImpl.value().notifyEventHandleActivity(mActiveEventHandles);

    CALLBACK_END_VOID
}

std::string VNCCommonDecoderPluginContextImpl::getSupportInfoDecoderName()
{
    if(!mImpl.hasValue())
    {
        return "<No decoder loaded>";
    }

    return mImpl.value().getSupportInfoDecoderName();
}

std::string VNCCommonDecoderPluginContextImpl::getSupportInfoDecoderVersion()
{
    if(!mImpl.hasValue())
    {
        return "<No decoder loaded>";
    }

    return mImpl.value().getSupportInfoDecoderVersion();
}

bool VNCCommonDecoderPluginContextImpl::getCreationParamDisableBuffering()
{
    if(!mImpl.hasValue())
    {
        throw std::logic_error("No decoder loaded");
    }

    return mImpl.value().getCreationParamDisableBuffering();
}



